import * as cheerio from 'cheerio'

const ATTR_ESCAPE_RE = /[&<>"']/g
const ATTR_ESCAPE_MAP: Record<string, string> = {
  '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', '\'': '&#39;',
}

export function escapeAttr (s: unknown) {
  return String(s).replace(ATTR_ESCAPE_RE, ch => ATTR_ESCAPE_MAP[ch])
}

export type Claim = {
  value: unknown // canonical numeric/string value from your DB
  country?: string
  date?: string | Date
}

export type Policy =
  | { type: 'exact' }
  | { type: 'rounded', decimals: number }
  | { type: 'tolerance', tolerance: number } // e.g. 0.02 = ±2%
  | { type: 'percent' }
  | { type: 'range' }
  | { type: 'abbr' } // K/M/B suffix handling
  | { type: 'ratio' } // “1 in N” or “1:N”
  | { type: 'year' } // compare only the year portion if dates
  | { type: 'auto' }

// ---------- formatting ----------
function toTitleAttr (country?: string, date?: string | Date, value?: string | number | undefined | null, policy?: Policy) {
  if (value === undefined) {
    return 'Claim not found'
  }
  const parts = [
    country ? `Country: ${country}` : '',
    date ? `Date: ${typeof date === 'string' ? date : date.toISOString().slice(0, 10)}` : '',
    value ? `Actual: ${value}` : '',
    policy ? `Policy: ${JSON.stringify(policy)}` : '',
  ].filter(Boolean)
  return escapeAttr(parts.join('\n')).replace(/\n/g, '&#10;')
}

function withMark (claimId: string, inner: string, ok: boolean, claim: Claim, policy: Policy) {
  const title = toTitleAttr(claim.country, claim.date, claim.value as string | number | undefined | null, policy)

  if (ok) {
    return `<claim id="${claimId}">${inner}<sup class="verified-mark" title="Verified data&#10;&#10;${title}">✓</sup></claim>`
  }

  return `<claim id="${claimId}">${inner}<sup class="verify-pending" title="Needs verification&#10;&#10;${title}" role="img" aria-label="Needs verification">⚠</sup></claim>`
}

// ---------- Small utilities ----------
const normalizeSpaces = (s: string) => s.replace(/\s+/g, ' ').trim()
const stripDecorations = (s: string) => s.replace(/[\u00A0 ,]/g, '')

const coalesceNumber = (...c: Array<number | null | undefined>): number => {
  for (const v of c) {
    if (v !== null && v !== undefined) {
      return v
    }
  }
  return Number.NaN
}

const isFiniteNumber = (n: unknown): n is number =>
  typeof n === 'number' && Number.isFinite(n)

// ---------- Parsers ----------
function parseAbbreviatedNumber (s: string): number | null {
  const m = s.trim().match(/^([+-]?\d+(?:\.\d+)?)([kmb])?$/i)
  if (!m) {
    return null
  }
  const n = Number(m[1])
  if (!Number.isFinite(n)) {
    return null
  }
  const unit = (m[2] || '').toLowerCase()
  const mult = unit === 'k' ? 1e3 : unit === 'm' ? 1e6 : unit === 'b' ? 1e9 : 1
  return n * mult
}

function parsePercentish (s: string): { value: number, isPercent: boolean } | null {
  const trimmed = s.trim()
  const percent = trimmed.endsWith('%')
  const numStr = percent ? trimmed.slice(0, -1) : trimmed
  const n = Number(numStr.replace(/[, ]/g, ''))
  if (!Number.isFinite(n)) {
    return null
  }
  return { value: percent ? n / 100 : n, isPercent: percent }
}

function parseRatio (s: string): number | null {
  const m = s.match(/^\s*(\d+(?:\.\d+)?)\s*(?:in|\/|:)\s*(\d+(?:\.\d+)?)\s*$/i)
  if (!m) {
    return null
  }
  const a = Number(m[1]), b = Number(m[2])
  if (!Number.isFinite(a) || !Number.isFinite(b) || b === 0) {
    return null
  }
  return a / b
}

function extractRange (s: string): { lo: number, hi: number } | null {
  const compact = normalizeSpaces(s.toLowerCase())
  const between = compact.match(/between\s+([^\s]+)\s+(?:and|to)\s+([^\s]+)/i)
  const simple = compact.match(/^([^\s]+)\s*(?:–|-|to)\s*([^\s]+)$/i)
  const m = between ?? simple
  if (!m) {
    return null
  }

  const aStr = stripDecorations(m[1]), bStr = stripDecorations(m[2])
  const a = parseAbbreviatedNumber(aStr) ?? Number(m[1])
  const b = parseAbbreviatedNumber(bStr) ?? Number(m[2])
  if (!Number.isFinite(a) || !Number.isFinite(b)) {
    return null
  }

  return { lo: Math.min(a, b), hi: Math.max(a, b) }
}

// ---------- Numeric builders (modular, linter-safe) ----------
function percentAwareNumber (raw: string): number {
  const cleaned = raw.replace(/[, ]/g, '')
  const hasPercent = raw.endsWith('%') // raw is string here
  const n = Number(cleaned.replace(/%$/, ''))
  if (!Number.isFinite(n)) {
    return Number.NaN
  }
  return hasPercent ? n / 100 : n
}

function buildNumericFromText (raw: string): {
  percentish: { value: number, isPercent: boolean } | null
  abbr: number | null
  ratio: number | null
  plain: number // percent-aware plain number
} {
  const percentish = parsePercentish(raw)
  const abbr = parseAbbreviatedNumber(stripDecorations(raw.toLowerCase()))
  const ratio = parseRatio(raw)
  const plain = percentAwareNumber(raw)
  return { percentish, abbr, ratio, plain }
}

// ---------- Comparators per policy ----------
const compareExact = (innerRaw: string, claimRaw: string) =>
  stripDecorations(innerRaw.replace(/%/g, ''))
  === stripDecorations(claimRaw.replace(/%/g, ''))

const comparePercent = (
  inner: { percentish: { value: number } | null },
  claim: { percentish: { value: number } | null },
) => !!inner.percentish && !!claim.percentish && inner.percentish.value === claim.percentish.value

const compareAbbr = (inner: { abbr: number | null }, claim: { abbr: number | null }) =>
  inner.abbr != null && claim.abbr != null && inner.abbr === claim.abbr

const compareRatio = (inner: { ratio: number | null }, claim: { ratio: number | null }) =>
  inner.ratio != null && claim.ratio != null && inner.ratio === claim.ratio

const compareRange = (innerRaw: string, claimNum: number) => {
  const range = extractRange(innerRaw)
  return !!range && isFiniteNumber(claimNum) && claimNum >= range.lo && claimNum <= range.hi
}

const numEqualRounded = (a: number, b: number, decimals: number) =>
  Number(a.toFixed(decimals)) === Number(b.toFixed(decimals))

const numWithinTolerance = (a: number, b: number, tol: number) => {
  if (a === 0 && b === 0) {
    return true
  }
  const denom = Math.max(1e-12, Math.max(Math.abs(a), Math.abs(b)))
  return Math.abs(a - b) / denom <= tol
}

const compareYear = (innerRaw: string, claim: Claim) => {
  const innerYear = Number(innerRaw.match(/\b(19|20)\d{2}\b/)?.[0])
  const date
      = typeof claim.date === 'string'
        ? new Date(claim.date)
        : (claim.date instanceof Date
            ? claim.date
            : new Date(String(claim.value)))
  const t = date.getTime()
  const claimYear = Number.isNaN(t) ? Number.NaN : new Date(t).getUTCFullYear()
  return Number.isFinite(innerYear) && Number.isFinite(claimYear) && innerYear === claimYear
}

// ---------- Dispatcher ----------
export function compareByPolicy (innerText: string, claim: Claim, policy: Policy): boolean {
  const innerRaw = String(normalizeSpaces(innerText).trim())
  const claimRaw = String(claim.value ?? '').trim()

  if (policy.type === 'exact' || policy.type === 'auto') {
    const exact = compareExact(innerRaw, claimRaw)
    if (policy.type === 'exact') {
      return exact
    } else {
      if (exact) {
        // If auto, and exact is true, return true
        // Otherwise, continue with the other policies
        return true
      }
    }
  }

  const inner = buildNumericFromText(innerRaw)
  const claimParts = buildNumericFromText(claimRaw)

  if (policy.type === 'percent') {
    return comparePercent(inner, claimParts)
  }
  if (policy.type === 'abbr') {
    return compareAbbr(inner, claimParts)
  }
  if (policy.type === 'ratio') {
    return compareRatio(inner, claimParts)
  }
  if (policy.type === 'year') {
    return compareYear(innerRaw, claim)
  }

  // Range uses inner text + a single numeric from claim
  if (policy.type === 'range') {
    const claimNum = coalesceNumber(
      claimParts.abbr,
      claimParts.percentish?.value,
      claimParts.ratio,
      claimParts.plain,
    )
    return compareRange(innerRaw, claimNum)
  }

  // Numeric policies
  const innerNum = coalesceNumber(inner.percentish?.value, inner.abbr, inner.ratio, inner.plain)
  const claimNum = coalesceNumber(claimParts.percentish?.value, claimParts.abbr, claimParts.ratio, claimParts.plain)
  const bothNumeric = Number.isFinite(innerNum) && Number.isFinite(claimNum)

  if (policy.type === 'rounded') {
    return bothNumeric && numEqualRounded(innerNum, claimNum, policy.decimals)
  }
  if (policy.type === 'tolerance') {
    return bothNumeric && numWithinTolerance(innerNum, claimNum, policy.tolerance)
  }

  // AUTO: try a sane order
  if (policy.type === 'auto') {
    if (compareExact(innerRaw, claimRaw)) {
      return true
    }
    if (comparePercent(inner, claimParts)) {
      return true
    }
    if (compareAbbr(inner, claimParts)) {
      return true
    }
    if (compareRatio(inner, claimParts)) {
      return true
    }
    if (bothNumeric && numEqualRounded(innerNum, claimNum, 0)) {
      return true
    }
    if (bothNumeric && numWithinTolerance(innerNum, claimNum, 0.02)) {
      return true
    }
    const claimForRange = coalesceNumber(claimNum)
    if (compareRange(innerRaw, claimForRange)) {
      return true
    }
    return false
  }

  // Fallback
  return false
}

// ---------- Public API ----------
export function pcnPolicy (
  claimId: string,
  cleanInner: string,
  claim: Claim,
  policy: Policy,
) {
  console.log('claimId', claimId)
  console.log('cleanInner', cleanInner)
  console.log('claim', claim)
  console.log('policy-cheerio', policy)
  // Determine policy

  const ok = compareByPolicy(cleanInner, claim, policy)
  return withMark(claimId, cleanInner, ok, claim, policy)
}

export function processPCN (claimId: string, innerText: string, claim: Claim, policy: Policy) {
  // Remove any existing verification markers to make this idempotent

  const cleanInner = innerText.replace(
    /<sup class="(?:verified-mark|verify-pending)".*?<\/sup>/g,
    '',
  ).replace(
    /<span class="needs-verify".*?>(.*?)<\/span>/g,
    '$1',
  )

  if (!claim) {
    // No known claim for this id → mark as pending
    return `<claim id="${claimId}">${cleanInner}<sup class="verify-pending" title="Needs verification" role="img" aria-label="Needs verification">⚠</sup></claim>`
  }

  return pcnPolicy(claimId, innerText, claim, policy)
}

export function processPCNClaims (content: string, claims: Record<string, Claim>) {
  // Parse the content with cheerio to get the claims and their policies
  const $ = cheerio.load(content, { xmlMode: false }) // tolerant of unquoted attrs

  const pcnClaims = $('claim').map((_, el) => ({
    tag: $(el).toString(),
    id: $(el).attr('id') ?? '',
    policy: {
    //   type: $(el).attr('policy') ?? 'auto',
      type: 'auto',
      decimals: $(el).attr('decimals') ? Number($(el).attr('decimals')) : undefined,
      tolerance: $(el).attr('tolerance') ? Number($(el).attr('tolerance')) : undefined,
    },
    value: $(el).text(),
  })).get()

  // Process the claims
  for (const claim of pcnClaims) {
    const replaceClaim = processPCN(claim.id, claim.value, claims?.[claim.id] ?? {}, claim.policy as Policy)
    content = content.replace(claim.tag, replaceClaim)
  }

  return content
}
